<?php
declare(strict_types=1);

/**
 * Company settings helpers (company_settings table)
 * Not: company_settings tablosunda company_id UNIQUE/PK olmalı.
 */
function settings_get(PDO $pdo, int $companyId): array {
  $st = $pdo->prepare("SELECT * FROM company_settings WHERE company_id=? LIMIT 1");
  $st->execute([$companyId]);
  $row = $st->fetch(PDO::FETCH_ASSOC);
  return $row ?: [];
}

function settings_columns(PDO $pdo): array {
  static $cols = null;
  if (is_array($cols)) return $cols;

  $cols = [];
  $st = $pdo->query("SHOW COLUMNS FROM company_settings");
  foreach ($st->fetchAll(PDO::FETCH_ASSOC) as $row) {
    $cols[] = (string)$row['Field'];
  }
  return $cols;
}

function settings_upsert(PDO $pdo, int $companyId, array $data): void {
  $cols = array_flip(settings_columns($pdo));

  $clean = [];
  foreach ($data as $k => $v) {
    if ($k === 'company_id') continue;
    if (!isset($cols[$k])) continue;
    $clean[$k] = $v;
  }
  if (!$clean) return;

  $fields = array_keys($clean);
  $insertCols = 'company_id,'.implode(',', $fields);
  $placeholders = implode(',', array_fill(0, count($fields)+1, '?')); // +company_id

  $updates = [];
  foreach ($fields as $f) $updates[] = "$f=VALUES($f)";

  $sql = "INSERT INTO company_settings ($insertCols) VALUES ($placeholders) ".
         "ON DUPLICATE KEY UPDATE ".implode(',', $updates);

  $params = [$companyId];
  foreach ($fields as $f) $params[] = $clean[$f];

  $st = $pdo->prepare($sql);
  $st->execute($params);
}

/*
|--------------------------------------------------------------------------
| Genel Render
|--------------------------------------------------------------------------
*/
function panel_render(string $page, array $data = []): void {
  $tenant = $GLOBALS['tenant'];
  $u = auth_user();

  // header/theme/lang için settings'i her sayfaya geçiriyoruz
  $pdo = $GLOBALS['pdo'];
  $cid = (int)($tenant['id'] ?? 0);
  $settings = $cid > 0 ? settings_get($pdo, $cid) : [];

  // default dil (session yoksa)
  if (!isset($_SESSION['lang']) || !is_string($_SESSION['lang']) || $_SESSION['lang'] === '') {
    $ld = (string)($settings['locale_default'] ?? 'tr');
    $_SESSION['lang'] = in_array($ld, ['tr','en'], true) ? $ld : 'tr';
  }

  view('layouts/panel', array_merge($data, [
    'tenant'   => $tenant,
    'user'     => $u,
    'page'     => $page,
    'settings' => $settings,
  ]));
}

/*
|--------------------------------------------------------------------------
| Dashboard
|--------------------------------------------------------------------------
*/
function panel_dashboard(): void {
  $pdo = $GLOBALS['pdo'];
  $cid = (int)$GLOBALS['tenant']['id'];

  // Customers
  $st = $pdo->prepare("SELECT COUNT(*) c FROM customers WHERE company_id=? AND deleted_at IS NULL");
  $st->execute([$cid]);
  $customersCount = (int)($st->fetch()['c'] ?? 0);

  // Products
  $st = $pdo->prepare("SELECT COUNT(*) c FROM products WHERE company_id=? AND deleted_at IS NULL");
  $st->execute([$cid]);
  $productsCount = (int)($st->fetch()['c'] ?? 0);

  // Open Invoices (draft + sent)
  $st = $pdo->prepare("
    SELECT COUNT(*) c
    FROM documents
    WHERE company_id=?
      AND type='invoice'
      AND deleted_at IS NULL
      AND status IN ('draft','sent')
  ");
  $st->execute([$cid]);
  $openInvoices = (int)($st->fetch()['c'] ?? 0);

  // Open Quotes (draft + sent)
  $st = $pdo->prepare("
    SELECT COUNT(*) c
    FROM documents
    WHERE company_id=?
      AND type='quote'
      AND deleted_at IS NULL
      AND status IN ('draft','sent')
  ");
  $st->execute([$cid]);
  $openQuotes = (int)($st->fetch()['c'] ?? 0);

  panel_render('dashboard', compact('customersCount','productsCount','openInvoices','openQuotes'));
}

/*
|--------------------------------------------------------------------------
| Customers
|--------------------------------------------------------------------------
*/
function panel_customers(): void {
  $pdo = $GLOBALS['pdo'];
  $cid = (int)$GLOBALS['tenant']['id'];

  $st = $pdo->prepare("
    SELECT * FROM customers
    WHERE company_id=? AND deleted_at IS NULL
    ORDER BY id DESC
  ");
  $st->execute([$cid]);
  $rows = $st->fetchAll();

  $ok = flash_get('ok');
  $err = flash_get('err');
  panel_render('customers', compact('rows','ok','err'));
}

function panel_customers_post(): void {
  $pdo = $GLOBALS['pdo'];
  $cid = (int)$GLOBALS['tenant']['id'];
  $slug = (string)$GLOBALS['tenant']['slug'];

  $csrf = (string)($_POST['csrf'] ?? '');
  if (!csrf_verify($csrf)) {
    flash_set('err', 'CSRF doğrulaması başarısız.');
    redirect(base_path("/{$slug}/customers"));
  }

  $action = (string)($_POST['action'] ?? 'create');

  if ($action === 'create') {
    $name = trim((string)($_POST['name'] ?? ''));
    $email = trim((string)($_POST['email'] ?? ''));
    $phone = trim((string)($_POST['phone'] ?? ''));
    $address = trim((string)($_POST['address'] ?? ''));
    $tax_office = trim((string)($_POST['tax_office'] ?? ''));
    $tax_no = trim((string)($_POST['tax_no'] ?? ''));

    if ($name === '') {
      flash_set('err', 'Müşteri adı zorunlu.');
      redirect(base_path("/{$slug}/customers"));
    }

    $st = $pdo->prepare("
      INSERT INTO customers (company_id,name,email,phone,address,tax_office,tax_no,is_active)
      VALUES (?,?,?,?,?,?,?,1)
    ");
    $st->execute([$cid,$name,$email ?: null,$phone ?: null,$address ?: null,$tax_office ?: null,$tax_no ?: null]);

    flash_set('ok', 'Müşteri eklendi.');
    redirect(base_path("/{$slug}/customers"));
  }

  if ($action === 'trash') {
    $id = (int)($_POST['id'] ?? 0);
    if ($id > 0) {
      $st = $pdo->prepare("UPDATE customers SET deleted_at=NOW() WHERE id=? AND company_id=?");
      $st->execute([$id,$cid]);
      flash_set('ok', 'Müşteri çöp kutusuna taşındı.');
    }
    redirect(base_path("/{$slug}/customers"));
  }

  flash_set('err', 'Bilinmeyen işlem.');
  redirect(base_path("/{$slug}/customers"));
}

/*
|--------------------------------------------------------------------------
| Products
|--------------------------------------------------------------------------
*/
function panel_products(): void {
  $pdo = $GLOBALS['pdo'];
  $cid = (int)$GLOBALS['tenant']['id'];

  $st = $pdo->prepare("
    SELECT * FROM products
    WHERE company_id=? AND deleted_at IS NULL
    ORDER BY id DESC
  ");
  $st->execute([$cid]);
  $rows = $st->fetchAll();

  $ok = flash_get('ok');
  $err = flash_get('err');
  panel_render('products', compact('rows','ok','err'));
}

function panel_products_post(): void {
  $pdo = $GLOBALS['pdo'];
  $cid = (int)$GLOBALS['tenant']['id'];
  $slug = (string)$GLOBALS['tenant']['slug'];

  $csrf = (string)($_POST['csrf'] ?? '');
  if (!csrf_verify($csrf)) {
    flash_set('err', 'CSRF doğrulaması başarısız.');
    redirect(base_path("/{$slug}/products"));
  }

  $action = (string)($_POST['action'] ?? 'create');

  if ($action === 'create') {
    $name = trim((string)($_POST['name'] ?? ''));
    $unit = trim((string)($_POST['unit'] ?? 'adet')) ?: 'adet';
    $price = (float)($_POST['price'] ?? 0);
    $vat = (float)($_POST['vat_rate'] ?? 20);
    $desc = trim((string)($_POST['description'] ?? ''));

    if ($name === '') {
      flash_set('err', 'Ürün/Hizmet adı zorunlu.');
      redirect(base_path("/{$slug}/products"));
    }

    $st = $pdo->prepare("
      INSERT INTO products (company_id,name,description,unit,price,vat_rate,is_active)
      VALUES (?,?,?,?,?,?,1)
    ");
    $st->execute([$cid,$name,$desc ?: null,$unit,$price,$vat]);

    flash_set('ok', 'Ürün/Hizmet eklendi.');
    redirect(base_path("/{$slug}/products"));
  }

  if ($action === 'trash') {
    $id = (int)($_POST['id'] ?? 0);
    if ($id > 0) {
      $st = $pdo->prepare("UPDATE products SET deleted_at=NOW() WHERE id=? AND company_id=?");
      $st->execute([$id,$cid]);
      flash_set('ok', 'Ürün/Hizmet çöp kutusuna taşındı.');
    }
    redirect(base_path("/{$slug}/products"));
  }

  flash_set('err', 'Bilinmeyen işlem.');
  redirect(base_path("/{$slug}/products"));
}

/*
|--------------------------------------------------------------------------
| Settings & Company
|--------------------------------------------------------------------------
*/
function panel_settings(): void {
  $pdo = $GLOBALS['pdo'];
  $cid = (int)$GLOBALS['tenant']['id'];
  $settings = settings_get($pdo, $cid);
  $ok = flash_get('ok');
  $err = flash_get('err');
  panel_render('settings', compact('settings','ok','err'));
}

function panel_settings_post(): void {
  $pdo = $GLOBALS['pdo'];
  $tenant = $GLOBALS['tenant'];
  $cid = (int)($tenant['id'] ?? 0);
  $slug = (string)($tenant['slug'] ?? tenant_slug());
  $tab  = (string)($_GET['tab'] ?? 'general');

  $csrf = (string)($_POST['csrf'] ?? '');
  if (function_exists('csrf_verify') && !csrf_verify($csrf)) {
    flash_set('err', 'CSRF doğrulaması başarısız.');
    header('Location: '.base_path('/'.$slug.'/settings?tab='.urlencode($tab)));
    exit;
  }

  $data = [];

  // Genel
  if (isset($_POST['default_currency'])) $data['default_currency'] = trim((string)$_POST['default_currency']);
  if (isset($_POST['default_currency_symbol'])) $data['default_currency_symbol'] = trim((string)$_POST['default_currency_symbol']);
  if (isset($_POST['default_vat_rate'])) {
    $vr = (float)$_POST['default_vat_rate'];
    if ($vr < 0) $vr = 0;
    if ($vr > 100) $vr = 100;
    $data['default_vat_rate'] = $vr;
  }
  if (isset($_POST['app_short'])) {
    $x = strtoupper(trim((string)$_POST['app_short']));
    if ($x === '') $x = 'FT';
    if (function_exists('mb_strlen') && mb_strlen($x, 'UTF-8') > 8) $x = mb_substr($x, 0, 8, 'UTF-8');
    $data['app_short'] = $x;
  }
  if (isset($_POST['app_name'])) {
    $x = trim((string)$_POST['app_name']);
    if ($x === '') $x = 'FaturaTakip';
    if (function_exists('mb_strlen') && mb_strlen($x, 'UTF-8') > 64) $x = mb_substr($x, 0, 64, 'UTF-8');
    $data['app_name'] = $x;
  }
  if (isset($_POST['after_create_redirect'])) {
    $acr = (string)$_POST['after_create_redirect'];
    if (!in_array($acr, ['view','print','list'], true)) $acr = 'view';
    $data['after_create_redirect'] = $acr;
  }
  if ($tab === 'general') $data['ui_show_slug'] = isset($_POST['ui_show_slug']) ? 1 : 0;

  // Belge No
  if (isset($_POST['invoice_prefix'])) $data['invoice_prefix'] = trim((string)$_POST['invoice_prefix']);
  if (isset($_POST['quote_prefix'])) $data['quote_prefix'] = trim((string)$_POST['quote_prefix']);
  if (isset($_POST['number_padding'])) {
    $p = (int)$_POST['number_padding'];
    if ($p < 1) $p = 1;
    if ($p > 12) $p = 12;
    $data['number_padding'] = $p;
  }
  if (isset($_POST['invoice_next_number'])) { $n = (int)$_POST['invoice_next_number']; if ($n < 1) $n = 1; $data['invoice_next_number'] = $n; }
  if (isset($_POST['quote_next_number'])) { $n = (int)$_POST['quote_next_number']; if ($n < 1) $n = 1; $data['quote_next_number'] = $n; }

  // Kargo
  if ($tab === 'shipping') $data['shipping_enabled'] = isset($_POST['shipping_enabled']) ? 1 : 0;
  if (isset($_POST['shipping_mode_default'])) {
    $sm = (string)$_POST['shipping_mode_default'];
    if (!in_array($sm, ['none','separate','line'], true)) $sm = 'separate';
    $data['shipping_mode_default'] = $sm;
  }
  if (isset($_POST['shipping_vat_rate'])) {
    $sv = (float)$_POST['shipping_vat_rate'];
    if ($sv < 0) $sv = 0;
    if ($sv > 100) $sv = 100;
    $data['shipping_vat_rate'] = $sv;
  }

  // PDF
  if (isset($_POST['pdf_template'])) $data['pdf_template'] = trim((string)$_POST['pdf_template']);
  if (isset($_POST['pdf_filename_pattern'])) $data['pdf_filename_pattern'] = trim((string)$_POST['pdf_filename_pattern']);

  // Tema
  if (isset($_POST['theme_mode'])) {
    $tm = (string)$_POST['theme_mode'];
    if (!in_array($tm, ['system','light','dark'], true)) $tm = 'system';
    $data['theme_mode'] = $tm;
  }
  if (isset($_POST['theme_primary'])) $data['theme_primary'] = trim((string)$_POST['theme_primary']);

  // Dil
  if (isset($_POST['locale_default'])) {
    $ld = (string)$_POST['locale_default'];
    if (!in_array($ld, ['tr','en'], true)) $ld = 'tr';
    $data['locale_default'] = $ld;
    $_SESSION['lang'] = $ld; // anında uygula
  }
  if (isset($_POST['locales_enabled'])) $data['locales_enabled'] = trim((string)$_POST['locales_enabled']);

  try {
    settings_upsert($pdo, $cid, $data);
  } catch (Throwable $e) {
    flash_set('err', 'Ayarlar kaydedilemedi: '.$e->getMessage());
    header('Location: '.base_path('/'.$slug.'/settings?tab='.urlencode($tab)));
    exit;
  }

  flash_set('ok', 'Ayarlar kaydedildi.');
  header('Location: '.base_path('/'.$slug.'/settings?tab='.urlencode($tab)));
  exit;
}

/**
 * ✅ COMPANY (GET) — Firma bilgilerini DB’den çekip company.php view’a gönderir
 */
function panel_company(): void {
  $pdo = $GLOBALS['pdo'];
  $tenant = $GLOBALS['tenant'];
  $cid = (int)($tenant['id'] ?? 0);

  $st = $pdo->prepare("SELECT * FROM companies WHERE id=? LIMIT 1");
  $st->execute([$cid]);
  $company = $st->fetch(PDO::FETCH_ASSOC) ?: [];

  $ok  = flash_get('ok');
  $err = flash_get('err');

  panel_render('company', compact('company','ok','err'));
}

/**
 * ✅ COMPANY (POST) — routes.php’in beklediği fonksiyon (hata buradan geliyordu)
 * - Firma bilgilerini günceller
 * - Logo yükler (uploads/{slug}/logo.ext)
 * - Logo kaldırır (action=remove_logo)
 */
function panel_company_post(): void {
  $pdo = $GLOBALS['pdo'];
  $tenant = $GLOBALS['tenant'];
  $cid = (int)($tenant['id'] ?? 0);
  $slug = (string)($tenant['slug'] ?? tenant_slug());

  // Auth + tenant güvenliği
  $u = require_auth();
  if ((int)($u['company_id'] ?? 0) !== $cid) {
    http_response_code(403);
    echo "Forbidden";
    exit;
  }

  // CSRF
  $csrf = (string)($_POST['csrf'] ?? '');
  if (!csrf_verify($csrf)) {
    flash_set('err', 'CSRF doğrulaması başarısız.');
    redirect(base_path("/{$slug}/company"));
  }

  $action = (string)($_POST['action'] ?? 'save');

  // Mevcut firmayı al (logo silme için path lazım)
  $st = $pdo->prepare("SELECT * FROM companies WHERE id=? LIMIT 1");
  $st->execute([$cid]);
  $cur = $st->fetch(PDO::FETCH_ASSOC) ?: [];

  // Logo kaldırma
  if ($action === 'remove_logo') {
    $logoPath = (string)($cur['logo_path'] ?? '');
    if ($logoPath !== '') {
      $projectRoot = dirname(__DIR__, 2);          // app/controllers -> proje kökü
      $publicRoot  = $projectRoot . '/public';
      $abs = $publicRoot . '/' . ltrim($logoPath, '/');
      if (is_file($abs)) @unlink($abs);
    }

    $pdo->prepare("UPDATE companies SET logo_path=NULL WHERE id=?")->execute([$cid]);
    flash_set('ok', 'Logo kaldırıldı.');
    redirect(base_path("/{$slug}/company"));
  }

  // Normal kayıt (firma bilgileri)
  $name = trim((string)($_POST['name'] ?? ''));
  $address = trim((string)($_POST['address'] ?? ''));
  $taxOffice = trim((string)($_POST['tax_office'] ?? ''));
  $taxNo = trim((string)($_POST['tax_no'] ?? ''));
  $phone = trim((string)($_POST['phone'] ?? ''));
  $email = trim((string)($_POST['email'] ?? ''));
  $website = trim((string)($_POST['website'] ?? ''));
  $iban = trim((string)($_POST['iban'] ?? ''));

  // (Opsiyonel) tema/dil alanları companies tablosunda var
  $themeMode = (string)($_POST['theme_mode'] ?? ($cur['theme_mode'] ?? 'system'));
  if (!in_array($themeMode, ['system','light','dark'], true)) $themeMode = 'system';

  $themePrimary = trim((string)($_POST['theme_primary'] ?? ($cur['theme_primary'] ?? '')));
  $locale = (string)($_POST['locale'] ?? ($cur['locale'] ?? 'tr'));
  if (!in_array($locale, ['tr','en'], true)) $locale = 'tr';

  if ($name === '') {
    flash_set('err', 'Firma adı zorunlu.');
    redirect(base_path("/{$slug}/company"));
  }

  // Logo upload (varsa)
  $newLogoPath = null;
  if (isset($_FILES['logo']) && is_array($_FILES['logo']) && (int)($_FILES['logo']['error'] ?? UPLOAD_ERR_NO_FILE) !== UPLOAD_ERR_NO_FILE) {
    $err = (int)($_FILES['logo']['error'] ?? UPLOAD_ERR_OK);
    if ($err !== UPLOAD_ERR_OK) {
      flash_set('err', 'Logo yükleme hatası (код='.$err.').');
      redirect(base_path("/{$slug}/company"));
    }

    $tmp = (string)($_FILES['logo']['tmp_name'] ?? '');
    $size = (int)($_FILES['logo']['size'] ?? 0);
    if ($size <= 0) {
      flash_set('err', 'Logo dosyası boş görünüyor.');
      redirect(base_path("/{$slug}/company"));
    }
    // 4MB limit
    if ($size > 4 * 1024 * 1024) {
      flash_set('err', 'Logo çok büyük (max 4MB).');
      redirect(base_path("/{$slug}/company"));
    }

    $orig = (string)($_FILES['logo']['name'] ?? '');
    $ext = strtolower(pathinfo($orig, PATHINFO_EXTENSION));
    $allowed = ['png','jpg','jpeg','webp'];
    if (!in_array($ext, $allowed, true)) {
      flash_set('err', 'Logo formatı desteklenmiyor. (png/jpg/jpeg/webp)');
      redirect(base_path("/{$slug}/company"));
    }

    $projectRoot = dirname(__DIR__, 2); // app/controllers -> proje kökü
    $publicRoot  = $projectRoot . '/public';
    $dir = $publicRoot . '/uploads/' . $slug;

    if (!is_dir($dir)) {
      @mkdir($dir, 0775, true);
    }
    if (!is_dir($dir)) {
      flash_set('err', 'Upload klasörü oluşturulamadı: uploads/'.$slug);
      redirect(base_path("/{$slug}/company"));
    }

    // Eski logoyu sil (varsa)
    $old = (string)($cur['logo_path'] ?? '');
    if ($old !== '') {
      $absOld = $publicRoot . '/' . ltrim($old, '/');
      if (is_file($absOld)) @unlink($absOld);
    }

    $filename = 'logo.' . $ext;
    $absDest = $dir . '/' . $filename;

    if (!@move_uploaded_file($tmp, $absDest)) {
      flash_set('err', 'Logo kaydedilemedi.');
      redirect(base_path("/{$slug}/company"));
    }

    $newLogoPath = 'uploads/' . $slug . '/' . $filename;
  }

  // Update companies
  $sql = "
    UPDATE companies
    SET name=?,
        address=?,
        tax_office=?,
        tax_no=?,
        phone=?,
        email=?,
        website=?,
        iban=?,
        theme_mode=?,
        theme_primary=?,
        locale=?"
        .($newLogoPath !== null ? ", logo_path=?" : "")."
    WHERE id=?
  ";

  $params = [
    $name,
    ($address !== '' ? $address : null),
    ($taxOffice !== '' ? $taxOffice : null),
    ($taxNo !== '' ? $taxNo : null),
    ($phone !== '' ? $phone : null),
    ($email !== '' ? $email : null),
    ($website !== '' ? $website : null),
    ($iban !== '' ? $iban : null),
    $themeMode,
    ($themePrimary !== '' ? $themePrimary : null),
    $locale,
  ];
  if ($newLogoPath !== null) $params[] = $newLogoPath;
  $params[] = $cid;

  try {
    $pdo->prepare($sql)->execute($params);
  } catch (Throwable $e) {
    flash_set('err', 'Firma bilgileri kaydedilemedi: '.$e->getMessage());
    redirect(base_path("/{$slug}/company"));
  }

  flash_set('ok', 'Firma bilgileri kaydedildi.');
  redirect(base_path("/{$slug}/company"));
}

/*
|--------------------------------------------------------------------------
| Documents - Create (GET)
|--------------------------------------------------------------------------
*/
function panel_create(): void {
  $pdo = $GLOBALS['pdo'];
  $cid = (int)$GLOBALS['tenant']['id'];

  $customers = $pdo->prepare("SELECT id,name FROM customers WHERE company_id=? AND deleted_at IS NULL AND is_active=1 ORDER BY name");
  $customers->execute([$cid]);
  $customers = $customers->fetchAll();

  $products = $pdo->prepare("SELECT id,name,description,unit,price,vat_rate FROM products WHERE company_id=? AND deleted_at IS NULL AND is_active=1 ORDER BY name");
  $products->execute([$cid]);
  $products = $products->fetchAll();

  $st = $pdo->prepare("SELECT default_currency, default_currency_symbol, default_vat_rate FROM company_settings WHERE company_id=? LIMIT 1");
  $st->execute([$cid]);
  $settings = $st->fetch() ?: ['default_currency'=>'TRY','default_currency_symbol'=>'₺','default_vat_rate'=>20];

  $ok = flash_get('ok');
  $err = flash_get('err');

  $docNoInvoice = next_doc_no($pdo, $cid, 'invoice');
  $docNoQuote   = next_doc_no($pdo, $cid, 'quote');

  panel_render('create', compact('customers','products','settings','docNoInvoice','docNoQuote','ok','err'));
}

/*
|--------------------------------------------------------------------------
| Documents - Create (POST)
|--------------------------------------------------------------------------
*/
function panel_create_post(): void {
  $pdo = $GLOBALS['pdo'];
  $cid = (int)$GLOBALS['tenant']['id'];
  $slug = (string)$GLOBALS['tenant']['slug'];

  $csrf = (string)($_POST['csrf'] ?? '');
  if (!csrf_verify($csrf)) {
    flash_set('err', 'CSRF doğrulaması başarısız.');
    redirect(base_path("/{$slug}/create"));
  }

  $type = (string)($_POST['type'] ?? 'invoice');
  if (!in_array($type, ['invoice','quote'], true)) $type = 'invoice';

  $customerId = (int)($_POST['customer_id'] ?? 0);
  $issueDate = (string)($_POST['issue_date'] ?? date('Y-m-d'));
  $dueDate = trim((string)($_POST['due_date'] ?? ''));
  $currency = (string)($_POST['currency'] ?? 'TRY');
  $currencySymbol = (string)($_POST['currency_symbol'] ?? '₺');
  $notes = trim((string)($_POST['notes'] ?? ''));

  $shippingMode = (string)($_POST['shipping_mode'] ?? 'none'); // none|separate|line
  if (!in_array($shippingMode, ['none','separate','line'], true)) $shippingMode = 'none';
  $shippingAmount = (float)($_POST['shipping_amount'] ?? 0);
  $shippingVat = (float)($_POST['shipping_vat_rate'] ?? 20);

  $items = $_POST['items'] ?? [];
  if (!is_array($items) || count($items) === 0) {
    flash_set('err', 'En az 1 kalem eklemelisin.');
    redirect(base_path("/{$slug}/create"));
  }
  if ($customerId <= 0) {
    flash_set('err', 'Müşteri seçmelisin.');
    redirect(base_path("/{$slug}/create"));
  }

  $docNo = trim((string)($_POST['doc_no'] ?? ''));
  if ($docNo === '') {
    $docNo = next_doc_no($pdo, $cid, $type);
  } else {
    if (!preg_match('/^[A-Za-z0-9][A-Za-z0-9\-_.\/]{1,31}$/', $docNo)) {
      flash_set('err', 'Belge No geçersiz. (Harf/rakam ve -_. / kullan)');
      redirect(base_path("/{$slug}/create"));
    }
    $st = $pdo->prepare("SELECT id FROM documents WHERE company_id=? AND type=? AND doc_no=? LIMIT 1");
    $st->execute([$cid,$type,$docNo]);
    if ($st->fetch()) {
      flash_set('err', 'Bu Belge No zaten kullanılmış.');
      redirect(base_path("/{$slug}/create"));
    }
  }

  $subtotal = 0.0; $vatTotal = 0.0; $grand = 0.0;
  $cleanItems = [];

  foreach ($items as $it) {
    if (!is_array($it)) continue;

    $name = trim((string)($it['name'] ?? ''));
    $desc = trim((string)($it['description'] ?? ''));
    $qty  = (float)($it['qty'] ?? 1);
    $unit = trim((string)($it['unit'] ?? 'adet')) ?: 'adet';
    $price= (float)($it['unit_price'] ?? 0);
    $vat  = (float)($it['vat_rate'] ?? 20);

    if ($name === '') continue;
    if ($qty <= 0) $qty = 1;

    $lineSub = money2($qty * $price);
    $lineVat = money2($lineSub * ($vat/100));
    $lineTot = money2($lineSub + $lineVat);

    $subtotal += $lineSub;
    $vatTotal += $lineVat;
    $grand    += $lineTot;

    $cleanItems[] = [
      'item_type' => 'product',
      'name' => $name,
      'description' => $desc ?: null,
      'qty' => $qty,
      'unit' => $unit,
      'unit_price' => $price,
      'vat_rate' => $vat,
      'line_subtotal' => $lineSub,
      'line_vat' => $lineVat,
      'line_total' => $lineTot
    ];
  }

  if (count($cleanItems) === 0) {
    flash_set('err', 'Geçerli kalem bulunamadı.');
    redirect(base_path("/{$slug}/create"));
  }

  if ($shippingMode === 'separate' && $shippingAmount > 0) {
    $shipSub = money2($shippingAmount);
    $shipVat = money2($shipSub * ($shippingVat/100));
    $shipTot = money2($shipSub + $shipVat);

    $subtotal += $shipSub;
    $vatTotal += $shipVat;
    $grand    += $shipTot;
  }

  $pdo->beginTransaction();
  try {
    $st = $pdo->prepare("
      INSERT INTO documents
      (company_id,type,doc_no,customer_id,currency,currency_symbol,issue_date,due_date,status,notes,
       shipping_mode,shipping_amount,shipping_vat_rate,subtotal,vat_total,total)
      VALUES (?,?,?,?,?,?,?,?, 'draft', ?, ?, ?, ?, ?, ?, ?)
    ");
    $st->execute([
      $cid, $type, $docNo, $customerId, $currency, $currencySymbol,
      $issueDate,
      ($dueDate !== '' ? $dueDate : null),
      ($notes !== '' ? $notes : null),
      $shippingMode, $shippingAmount, $shippingVat,
      money2($subtotal), money2($vatTotal), money2($grand)
    ]);
    $docId = (int)$pdo->lastInsertId();

    $stItem = $pdo->prepare("
      INSERT INTO document_items
      (company_id,document_id,item_type,name,description,qty,unit,unit_price,vat_rate,line_subtotal,line_vat,line_total)
      VALUES (?,?,?,?,?,?,?,?,?,?,?,?)
    ");

    foreach ($cleanItems as $ci) {
      $stItem->execute([
        $cid,$docId,$ci['item_type'],$ci['name'],$ci['description'],$ci['qty'],$ci['unit'],
        $ci['unit_price'],$ci['vat_rate'],$ci['line_subtotal'],$ci['line_vat'],$ci['line_total']
      ]);
    }

    if ($shippingMode === 'line' && $shippingAmount > 0) {
      $shipSub = money2($shippingAmount);
      $shipVat = money2($shipSub * ($shippingVat/100));
      $shipTot = money2($shipSub + $shipVat);

      $stItem->execute([
        $cid,$docId,'shipping','Kargo',null,1,'', $shippingAmount, $shippingVat, $shipSub, $shipVat, $shipTot
      ]);
    }

    $pdo->commit();
  } catch (Throwable $e) {
    $pdo->rollBack();
    flash_set('err', 'Kayıt hatası: '.$e->getMessage());
    redirect(base_path("/{$slug}/create"));
  }

  flash_set('ok', ($type === 'invoice' ? 'Fatura' : 'Teklif') . " oluşturuldu: {$docNo}");
  redirect(base_path("/{$slug}/" . ($type === 'invoice' ? 'invoices' : 'quotes')));
}

/*
|--------------------------------------------------------------------------
| Documents - Lists
|--------------------------------------------------------------------------
*/
function panel_invoices(): void { panel_documents_list('invoice'); }
function panel_quotes(): void { panel_documents_list('quote'); }

/*
|--------------------------------------------------------------------------
| Invoice: View / Edit / Print
|--------------------------------------------------------------------------
*/
function fetch_document_with_items(PDO $pdo, int $cid, string $type, int $id): array {
  $st = $pdo->prepare("
    SELECT d.*, c.name AS customer_name, c.email AS customer_email, c.phone AS customer_phone, c.address AS customer_address,
           c.tax_office AS customer_tax_office, c.tax_no AS customer_tax_no
    FROM documents d
    JOIN customers c ON c.id=d.customer_id
    WHERE d.company_id=? AND d.type=? AND d.id=? AND d.deleted_at IS NULL
    LIMIT 1
  ");
  $st->execute([$cid,$type,$id]);
  $doc = $st->fetch(PDO::FETCH_ASSOC);
  if (!$doc) return [];

  $st = $pdo->prepare("
    SELECT * FROM document_items
    WHERE company_id=? AND document_id=?
    ORDER BY id ASC
  ");
  $st->execute([$cid,$id]);
  $items = $st->fetchAll(PDO::FETCH_ASSOC);

  return ['doc'=>$doc,'items'=>$items];
}

function panel_invoice_view(): void {
  $pdo = $GLOBALS['pdo'];
  $cid = (int)$GLOBALS['tenant']['id'];
  $id  = (int)($_GET['id'] ?? 0);

  $data = fetch_document_with_items($pdo, $cid, 'invoice', $id);
  if (!$data) { http_response_code(404); echo "Document not found"; return; }

  $ok = flash_get('ok');
  $err = flash_get('err');
  $doc = $data['doc'];
  $items = $data['items'];

  panel_render('invoice_view', compact('doc','items','ok','err'));
}

function panel_invoice_print(): void {
  $pdo = $GLOBALS['pdo'];
  $cid = (int)$GLOBALS['tenant']['id'];
  $id  = (int)($_GET['id'] ?? 0);

  $data = fetch_document_with_items($pdo, $cid, 'invoice', $id);
  if (!$data) { http_response_code(404); echo "Document not found"; return; }

  $doc = $data['doc'];
  $items = $data['items'];

  $st = $pdo->prepare("SELECT * FROM companies WHERE id=? LIMIT 1");
  $st->execute([$cid]);
  $company = $st->fetch(PDO::FETCH_ASSOC) ?: ['name'=>'Firma'];

  panel_render('invoice_print', compact('doc','items','company'));
}

function panel_invoice_edit(): void {
  $pdo = $GLOBALS['pdo'];
  $cid = (int)$GLOBALS['tenant']['id'];
  $id  = (int)($_GET['id'] ?? 0);

  $data = fetch_document_with_items($pdo, $cid, 'invoice', $id);
  if (!$data) { http_response_code(404); echo "Document not found"; return; }

  $customers = $pdo->prepare("SELECT id,name FROM customers WHERE company_id=? AND deleted_at IS NULL AND is_active=1 ORDER BY name");
  $customers->execute([$cid]);
  $customers = $customers->fetchAll(PDO::FETCH_ASSOC);

  $products = $pdo->prepare("SELECT id,name,description,unit,price,vat_rate FROM products WHERE company_id=? AND deleted_at IS NULL AND is_active=1 ORDER BY name");
  $products->execute([$cid]);
  $products = $products->fetchAll(PDO::FETCH_ASSOC);

  $st = $pdo->prepare("SELECT default_currency, default_currency_symbol, default_vat_rate FROM company_settings WHERE company_id=? LIMIT 1");
  $st->execute([$cid]);
  $settings = $st->fetch(PDO::FETCH_ASSOC) ?: ['default_currency'=>'TRY','default_currency_symbol'=>'₺','default_vat_rate'=>20];

  $ok = flash_get('ok');
  $err = flash_get('err');
  $doc = $data['doc'];
  $items = $data['items'];

  panel_render('invoice_edit', compact('doc','items','customers','products','settings','ok','err'));
}

function panel_invoice_edit_post(): void {
  $pdo = $GLOBALS['pdo'];
  $cid = (int)$GLOBALS['tenant']['id'];
  $slug = (string)$GLOBALS['tenant']['slug'];
  $id  = (int)($_GET['id'] ?? $_POST['id'] ?? 0);
  if ($id <= 0) redirect(base_path("/{$slug}/invoices"));

  $csrf = (string)($_POST['csrf'] ?? '');
  if (!csrf_verify($csrf)) {
    flash_set('err', 'CSRF doğrulaması başarısız.');
    redirect(base_path("/{$slug}/invoices/edit?id={$id}"));
  }

  $data = fetch_document_with_items($pdo, $cid, 'invoice', $id);
  if (!$data) {
    flash_set('err', 'Fatura bulunamadı.');
    redirect(base_path("/{$slug}/invoices"));
  }

  $customerId = (int)($_POST['customer_id'] ?? 0);
  $issueDate = (string)($_POST['issue_date'] ?? date('Y-m-d'));
  $dueDate = trim((string)($_POST['due_date'] ?? ''));
  $currency = (string)($_POST['currency'] ?? (string)($data['doc']['currency'] ?? 'TRY'));
  $currencySymbol = (string)($_POST['currency_symbol'] ?? (string)($data['doc']['currency_symbol'] ?? '₺'));
  $status = (string)($_POST['status'] ?? (string)($data['doc']['status'] ?? 'draft'));
  if (!in_array($status, ['draft','sent','paid','overdue'], true)) $status = 'draft';

  $notes = trim((string)($_POST['notes'] ?? ''));

  $shippingMode = (string)($_POST['shipping_mode'] ?? 'none');
  if (!in_array($shippingMode, ['none','separate','line'], true)) $shippingMode = 'none';
  $shippingAmount = (float)($_POST['shipping_amount'] ?? 0);
  $shippingVat = (float)($_POST['shipping_vat_rate'] ?? 20);

  $docNo = trim((string)($_POST['doc_no'] ?? (string)$data['doc']['doc_no']));
  if ($docNo === '' || !preg_match('/^[A-Za-z0-9][A-Za-z0-9\-_.\/]{1,31}$/', $docNo)) {
    flash_set('err', 'Belge No geçersiz.');
    redirect(base_path("/{$slug}/invoices/edit?id={$id}"));
  }

  $st = $pdo->prepare("SELECT id FROM documents WHERE company_id=? AND type='invoice' AND doc_no=? AND id<>? LIMIT 1");
  $st->execute([$cid,$docNo,$id]);
  if ($st->fetch()) {
    flash_set('err', 'Bu Belge No başka bir faturada kullanılıyor.');
    redirect(base_path("/{$slug}/invoices/edit?id={$id}"));
  }

  if ($customerId <= 0) {
    flash_set('err', 'Müşteri seçmelisin.');
    redirect(base_path("/{$slug}/invoices/edit?id={$id}"));
  }

  $itemsIn = $_POST['items'] ?? [];
  if (!is_array($itemsIn) || count($itemsIn) === 0) {
    flash_set('err', 'En az 1 kalem eklemelisin.');
    redirect(base_path("/{$slug}/invoices/edit?id={$id}"));
  }

  $subtotal = 0.0; $vatTotal = 0.0; $grand = 0.0;
  $cleanItems = [];

  foreach ($itemsIn as $it) {
    if (!is_array($it)) continue;
    $name = trim((string)($it['name'] ?? ''));
    $desc = trim((string)($it['description'] ?? ''));
    $qty  = (float)($it['qty'] ?? 1);
    $unit = trim((string)($it['unit'] ?? 'adet')) ?: 'adet';
    $price= (float)($it['unit_price'] ?? 0);
    $vat  = (float)($it['vat_rate'] ?? 20);
    if ($name === '') continue;
    if ($qty <= 0) $qty = 1;

    $lineSub = money2($qty * $price);
    $lineVat = money2($lineSub * ($vat/100));
    $lineTot = money2($lineSub + $lineVat);

    $subtotal += $lineSub;
    $vatTotal += $lineVat;
    $grand    += $lineTot;

    $cleanItems[] = [
      'item_type'=>'product',
      'name'=>$name,
      'description'=>$desc ?: null,
      'qty'=>$qty,
      'unit'=>$unit,
      'unit_price'=>$price,
      'vat_rate'=>$vat,
      'line_subtotal'=>$lineSub,
      'line_vat'=>$lineVat,
      'line_total'=>$lineTot,
    ];
  }

  if (count($cleanItems) === 0) {
    flash_set('err', 'Geçerli kalem bulunamadı.');
    redirect(base_path("/{$slug}/invoices/edit?id={$id}"));
  }

  if ($shippingMode === 'separate' && $shippingAmount > 0) {
    $shipSub = money2($shippingAmount);
    $shipVat = money2($shipSub * ($shippingVat/100));
    $shipTot = money2($shipSub + $shipVat);
    $subtotal += $shipSub;
    $vatTotal += $shipVat;
    $grand    += $shipTot;
  }

  $pdo->beginTransaction();
  try {
    $st = $pdo->prepare("
      UPDATE documents
      SET doc_no=?, customer_id=?, currency=?, currency_symbol=?, issue_date=?, due_date=?, status=?, notes=?,
          shipping_mode=?, shipping_amount=?, shipping_vat_rate=?, subtotal=?, vat_total=?, total=?
      WHERE id=? AND company_id=? AND type='invoice'
    ");
    $st->execute([
      $docNo,$customerId,$currency,$currencySymbol,$issueDate,($dueDate!==''?$dueDate:null),$status,($notes!==''?$notes:null),
      $shippingMode,$shippingAmount,$shippingVat,money2($subtotal),money2($vatTotal),money2($grand),
      $id,$cid
    ]);

    $pdo->prepare("DELETE FROM document_items WHERE company_id=? AND document_id=?")->execute([$cid,$id]);
    $stItem = $pdo->prepare("
      INSERT INTO document_items
      (company_id,document_id,item_type,name,description,qty,unit,unit_price,vat_rate,line_subtotal,line_vat,line_total)
      VALUES (?,?,?,?,?,?,?,?,?,?,?,?)
    ");
    foreach ($cleanItems as $ci) {
      $stItem->execute([
        $cid,$id,$ci['item_type'],$ci['name'],$ci['description'],$ci['qty'],$ci['unit'],
        $ci['unit_price'],$ci['vat_rate'],$ci['line_subtotal'],$ci['line_vat'],$ci['line_total']
      ]);
    }

    if ($shippingMode === 'line' && $shippingAmount > 0) {
      $shipSub = money2($shippingAmount);
      $shipVat = money2($shipSub * ($shippingVat/100));
      $shipTot = money2($shipSub + $shipVat);
      $stItem->execute([$cid,$id,'shipping','Kargo',null,1,'',$shippingAmount,$shippingVat,$shipSub,$shipVat,$shipTot]);
    }

    $pdo->commit();
  } catch (Throwable $e) {
    $pdo->rollBack();
    flash_set('err', 'Kayıt hatası: '.$e->getMessage());
    redirect(base_path("/{$slug}/invoices/edit?id={$id}"));
  }

  flash_set('ok', 'Fatura güncellendi.');
  redirect(base_path("/{$slug}/invoices/view?id={$id}"));
}

function panel_documents_list(string $type): void {
  $pdo = $GLOBALS['pdo'];
  $cid = (int)$GLOBALS['tenant']['id'];

  $st = $pdo->prepare("
    SELECT d.id,d.doc_no,d.issue_date,d.due_date,d.total,d.currency_symbol,d.status,c.name AS customer_name
    FROM documents d
    JOIN customers c ON c.id=d.customer_id
    WHERE d.company_id=? AND d.type=? AND d.deleted_at IS NULL
    ORDER BY d.id DESC
  ");
  $st->execute([$cid,$type]);
  $rows = $st->fetchAll(PDO::FETCH_ASSOC);

  $ok = flash_get('ok');
  $err = flash_get('err');

  panel_render($type === 'invoice' ? 'invoices' : 'quotes', compact('rows','ok','err','type'));
}

function panel_documents_post(): void {
  $pdo = $GLOBALS['pdo'];
  $cid = (int)$GLOBALS['tenant']['id'];
  $slug = (string)$GLOBALS['tenant']['slug'];

  $csrf = (string)($_POST['csrf'] ?? '');
  if (!csrf_verify($csrf)) {
    flash_set('err', 'CSRF doğrulaması başarısız.');
    redirect(base_path("/{$slug}/dashboard"));
  }

  $action = (string)($_POST['action'] ?? '');
  $id = (int)($_POST['id'] ?? 0);
  $type = (string)($_POST['type'] ?? 'invoice');

  if ($id <= 0) redirect(base_path("/{$slug}/dashboard"));
  if (!in_array($type, ['invoice','quote'], true)) $type = 'invoice';

  if ($action === 'trash') {
    $st = $pdo->prepare("UPDATE documents SET deleted_at=NOW() WHERE id=? AND company_id=? AND type=?");
    $st->execute([$id,$cid,$type]);
    flash_set('ok', 'Belge çöp kutusuna taşındı.');
    redirect(base_path("/{$slug}/" . ($type==='invoice'?'invoices':'quotes')));
  }

  flash_set('err', 'Bilinmeyen işlem.');
  redirect(base_path("/{$slug}/" . ($type==='invoice'?'invoices':'quotes')));
}

/*
|--------------------------------------------------------------------------
| Trash (Documents)  ✅ TEK SÜRÜM
|--------------------------------------------------------------------------
*/
function panel_trash(): void {
  $pdo = $GLOBALS['pdo'];
  $tenant = $GLOBALS['tenant'];
  $cid = (int)$tenant['id'];

  $type = $_GET['type'] ?? 'all'; // invoice | quote | all

  $sql = "
    SELECT d.*
    FROM documents d
    WHERE d.company_id = ?
      AND d.deleted_at IS NOT NULL
  ";
  $params = [$cid];

  if ($type === 'invoice') {
    $sql .= " AND d.type = 'invoice'";
  } elseif ($type === 'quote') {
    $sql .= " AND d.type = 'quote'";
  }

  $sql .= " ORDER BY d.deleted_at DESC";

  $st = $pdo->prepare($sql);
  $st->execute($params);
  $rows = $st->fetchAll(PDO::FETCH_ASSOC);

  $ok  = flash_get('ok');
  $err = flash_get('err');

  panel_render('trash', compact('rows','type','ok','err'));
}

function panel_trash_restore_post(): void {
  $pdo = $GLOBALS['pdo'];
  $tenant = $GLOBALS['tenant'];
  $cid = (int)$tenant['id'];

  csrf_or_fail(); // parametresiz artık sorun değil

  $ids = $_POST['ids'] ?? [];
  if (!is_array($ids) || !$ids) redirect_back();

  $ids = array_values(array_filter(array_map('intval', $ids), fn($x)=>$x>0));
  if (!$ids) redirect_back();

  $in = implode(',', array_fill(0, count($ids), '?'));
  $sql = "UPDATE documents SET deleted_at=NULL WHERE company_id=? AND id IN ($in)";
  $pdo->prepare($sql)->execute(array_merge([$cid], $ids));

  flash_set('ok', 'Seçilen belgeler geri alındı.');
  redirect_back();
}

function panel_trash_delete_post(): void {
  $pdo = $GLOBALS['pdo'];
  $tenant = $GLOBALS['tenant'];
  $cid = (int)$tenant['id'];

  csrf_or_fail(); // parametresiz artık sorun değil

  $ids = $_POST['ids'] ?? [];
  if (!is_array($ids) || !$ids) redirect_back();

  $ids = array_values(array_filter(array_map('intval', $ids), fn($x)=>$x>0));
  if (!$ids) redirect_back();

  $in = implode(',', array_fill(0, count($ids), '?'));

  $pdo->prepare("DELETE FROM document_items WHERE company_id=? AND document_id IN ($in)")
      ->execute(array_merge([$cid], $ids));

  $pdo->prepare("DELETE FROM documents WHERE company_id=? AND id IN ($in)")
      ->execute(array_merge([$cid], $ids));

  flash_set('ok', 'Seçilen belgeler kalıcı olarak silindi.');
  redirect_back();
}

/*
|--------------------------------------------------------------------------
| Helpers (en altta)
|--------------------------------------------------------------------------
*/
function csrf_or_fail(?string $redirectTo = null): void {
  $csrf = (string)($_POST['csrf'] ?? '');
  if (!function_exists('csrf_verify') || !csrf_verify($csrf)) {
    flash_set('err', 'CSRF doğrulaması başarısız.');

    if ($redirectTo) {
      redirect($redirectTo);
    }

    if (function_exists('redirect_back')) {
      redirect_back();
    }

    $ref = (string)($_SERVER['HTTP_REFERER'] ?? '');
    if ($ref !== '') {
      header('Location: '.$ref);
      exit;
    }

    $slug = (string)($GLOBALS['tenant']['slug'] ?? tenant_slug());
    redirect(base_path('/'.$slug.'/dashboard'));
  }
}

/* Profil */
function users_columns(PDO $pdo): array {
  static $cols = null;
  if ($cols !== null) return $cols;
  $cols = [];
  $st = $pdo->query("SHOW COLUMNS FROM users");
  foreach ($st->fetchAll(PDO::FETCH_ASSOC) as $r) $cols[] = (string)$r['Field'];
  return $cols;
}

function panel_profile(): void {
  $pdo    = $GLOBALS['pdo'];
  $tenant = $GLOBALS['tenant'];
  $u      = require_auth();

  if ((int)$u['company_id'] !== (int)$tenant['id']) {
    http_response_code(403); echo "Forbidden"; exit;
  }

  $cols = users_columns($pdo);

  // username yoksa select'e koyma
  $fields = ['id','company_id','name','email','role','is_active','created_at'];
  if (in_array('locale', $cols, true)) $fields[] = 'locale';
  if (in_array('theme_mode', $cols, true)) $fields[] = 'theme_mode';

  $fields = array_values(array_filter($fields, fn($f)=>in_array($f, $cols, true)));

  $sql = "SELECT ".implode(',', $fields)." FROM users WHERE id=? LIMIT 1";
  $st = $pdo->prepare($sql);
  $st->execute([(int)$u['id']]);
  $user = $st->fetch(PDO::FETCH_ASSOC) ?: $u;

  $ok = flash_get('ok');
  $err = flash_get('err');

  panel_render('profile', compact('user','ok','err'));
}

function panel_profile_post(): void {
  $pdo    = $GLOBALS['pdo'];
  $tenant = $GLOBALS['tenant'];
  $slug   = (string)$tenant['slug'];
  $u      = require_auth();

  if ((int)$u['company_id'] !== (int)$tenant['id']) {
    http_response_code(403); echo "Forbidden"; exit;
  }

  csrf_or_fail(base_path('/'.$slug.'/profile'));

  $cols = users_columns($pdo);

  $name  = trim((string)($_POST['name'] ?? ''));
  $email = trim((string)($_POST['email'] ?? ''));

  $curPass = (string)($_POST['current_password'] ?? '');
  $newPass = (string)($_POST['new_password'] ?? '');
  $newPass2= (string)($_POST['new_password2'] ?? '');

  $passCol = in_array('password_hash', $cols, true) ? 'password_hash' : (in_array('password', $cols, true) ? 'password' : null);
  if (!$passCol) {
    flash_set('err', 'users tablosunda şifre kolonu bulunamadı (password_hash/password).');
    redirect(base_path('/'.$slug.'/profile'));
  }

  $st = $pdo->prepare("SELECT id, company_id, {$passCol} AS pass FROM users WHERE id=? LIMIT 1");
  $st->execute([(int)$u['id']]);
  $row = $st->fetch(PDO::FETCH_ASSOC);

  if (!$row || (int)$row['company_id'] !== (int)$tenant['id']) {
    flash_set('err','Kullanıcı bulunamadı.');
    redirect(base_path('/'.$slug.'/profile'));
  }

  $pdo->beginTransaction();
  try {
    $set = [];
    $vals = [];

    if (in_array('name', $cols, true))  { $set[] = "name=?";  $vals[] = ($name !== '' ? $name : null); }
    if (in_array('email', $cols, true)) { $set[] = "email=?"; $vals[] = ($email !== '' ? $email : null); }

    if ($set) {
      $vals[] = (int)$u['id'];
      $pdo->prepare("UPDATE users SET ".implode(',', $set)." WHERE id=?")->execute($vals);
    }

    if ($curPass !== '' || $newPass !== '' || $newPass2 !== '') {
      if (!password_verify($curPass, (string)$row['pass'])) {
        throw new Exception("Mevcut şifre yanlış.");
      }
      if (strlen($newPass) < 6) {
        throw new Exception("Yeni şifre en az 6 karakter olmalı.");
      }
      if ($newPass !== $newPass2) {
        throw new Exception("Yeni şifre tekrarı eşleşmiyor.");
      }
      $hash = password_hash($newPass, PASSWORD_DEFAULT);
      $pdo->prepare("UPDATE users SET {$passCol}=? WHERE id=?")->execute([$hash, (int)$u['id']]);
    }

    $pdo->commit();
    flash_set('ok','Kullanıcı ayarları kaydedildi.');
  } catch (Throwable $e) {
    $pdo->rollBack();
    flash_set('err', $e->getMessage());
  }

  redirect(base_path('/'.$slug.'/profile'));
}
